// Author: https://makerworld.com/@LayerCake
// Link: https://makerworld.com/en/models/476846
// License: MIT

// Based on the excellent Gridfinity system by Zach Freedman
// Wiki: https://gridfinity.xyz/

// Differeces from the standard design:
//  - The interface has an added 0.4 mm flat section on top.
//  - The bottom chamfer has been extended for improved print bed adhesion.
//  - The base units can be overwritten which breaks compatibility but can use all space available.

/* [Gridfinity Units] */
// One Width Unit equals 42 mm
Units_Wide = 4;
// One Depth Unit equals 42 mm
Units_Deep = 4;

/* [Extra Margins] */
// Default: 0 mm
Margin_Left = 0;
// Default: 0 mm
Margin_Top = 0;
// Default: 0 mm
Margin_Right = 0;
// Default: 0 mm
Margin_Bottom = 0;

/* [Rounded Corners] */
// Default: 4 mm
Radius_Front_Left = 4;
// Default: 4 mm
Radius_Back_Left = 4;
// Default: 4 mm
Radius_Back_Right = 4;
// Default: 4 mm
Radius_Front_Right = 4;

/* [Compatibility Breaking Settings] */
// Default: 42 mm
Base_Unit_Width_Depth = 42;
// Default: 4 mm (Minimum: 4 mm)
Base_Unit_Radius = 4;

/* [Hidden] */
$fn = 200;

// Base Unit Width/Depth
b_xy = Base_Unit_Width_Depth;
// Base Radius
b_r = Base_Unit_Radius;

// Interface Top Flat
i_t_f = 0.4;
// Interface Top Chamfer
i_t_c = 1.75;
// Interface Middle Height
i_m_h = 1.8;
// Interface Bottom Chamfer
i_b_c = 1.05;
// Interface Overall Height
i_h = i_t_c + i_m_h + i_b_c;

GridfinityBasePlate();

module GridfinityBasePlate()
{
    difference() {
        // Grid
        g_w = b_xy * Units_Wide;
        g_d = b_xy * Units_Deep;
        // Main Body
        mb_w = g_w + Margin_Left + Margin_Right;
        mb_d = g_d + Margin_Top + Margin_Bottom;
        translate([-g_w/2-Margin_Left,-g_d/2-Margin_Bottom,0]) {
            difference() {
                cube([mb_w, mb_d, i_h]);
                CornerChamferCutter(Radius_Front_Left);
                translate([0,mb_d,0]) {
                    mirror([0,1,0]) {
                        CornerChamferCutter(Radius_Back_Left);
                    }
                }
                translate([mb_w,mb_d,0]) {
                    mirror([1,1,0]) {
                        CornerChamferCutter(Radius_Back_Right);
                    }
                }
                translate([mb_w,0,0]) {
                    mirror([1,0,0]) {
                        CornerChamferCutter(Radius_Front_Right);
                    }
                }
            }
        }
        // Interface Pattern
        translate([-b_xy*(Units_Wide/2-0.5),-b_xy*(Units_Deep/2-0.5),0]) {
            for(w=[1:Units_Wide]) {
                for(d=[1:Units_Deep]) {
                    translate([b_xy*(w-1),b_xy*(d-1),0]) {
                        GridfinityInterface();
                    }
                }
            }
        }
    }
}

module GridfinityInterface() {
    union() {
        translate([0,0,-0.5]) {
            BottomChamferedRoundedSquare(b_xy-(i_t_f+i_t_c)*2, b_xy-(i_t_f+i_t_c)*2, i_b_c+i_m_h+1.5, b_r-(i_t_f+i_t_c), i_b_c+0.5);
        }
        translate([0,0,i_b_c+i_m_h]) {
            BottomChamferedRoundedSquare(b_xy-i_t_f, b_xy-i_t_f, i_t_c+0.5, b_r-i_t_f, i_t_c+0.5);
        }
    }
}

module CornerChamferCutter(radius) {
    translate([radius,radius,0]) {
        mirror([1,1,0]) {
            difference() {
                translate([0,0,-1]) {
                    cube([radius*2, radius*2, i_h+2]);
                }
                translate([0,0,-2]) {
                    cylinder(h=i_h+4,r=radius);
                }
            }
        }
    }
}

module RoundedSquare(width, depth, height, radius) {
    minkowski() {
        rs_x = width - radius * 2;
        rs_y = depth - radius * 2;
        XYCenteredCube(rs_x, rs_y, height/2);
        cylinder(r=radius, h=height/2);
    }
}

module BottomChamferedRoundedSquare(width, depth, height, radius, chamfer){
    union() {
        difference() {
            // larger/straight dimensions
            l_x = width/2;
            l_y = depth/2;
            // smaller/chamfered dimensions
            s_x = l_x - chamfer;
            s_y = l_y - chamfer;
            polyhedron(
                points=[[s_x,s_y,0], [s_x,-s_y,0], [-s_x,-s_y,0], [-s_x,l_y-chamfer,0], // base
                        [l_x,l_y,chamfer], [l_x,-l_y,chamfer], [-l_x,-l_y,chamfer], [-l_x,l_y,chamfer], //middle
                        [l_x,l_y,height], [l_x,-l_y,height], [-l_x,-l_y,height], [-l_x,l_y,height]], //top 
                faces=[[3,2,1,0], // base
                       [5,4,0,1], [4,7,3,0], [7,6,2,3], [6,5,1,2], // angled sides
                       [9,8,4,5], [8,11,7,4], [11,10,6,7], [10,9,5,6], // straight sides
                       [11,8,9,10]] // top
            );
            // Cutout Corners
            rcs_c1x = width / 2 - radius / 2 + 1;
            rcs_c1y = depth / 2 - radius / 2 + 1;
            translate([rcs_c1x,rcs_c1y,height/2]) {
                cube([radius+2, radius+2, height+1], true);
            }
            translate([rcs_c1x,-rcs_c1y,height/2]) {
                cube([radius+2, radius+2, height+1], true);
            }
            translate([-rcs_c1x,-rcs_c1y,height/2]) {
                cube([radius+2, radius+2, height+1], true);
            }
            translate([-rcs_c1x,rcs_c1y,height/2]) {
                cube([radius+2, radius+2, height+1], true);
            }
        }
        // Rounded Corners
        rcs_c2x = width / 2 - radius;
        rcs_c2y = depth / 2 - radius;
        // Chamfer
        translate([rcs_c2x,rcs_c2y,0]) {
            BottomChamferedCylinder(radius, height, chamfer);
        }
        translate([rcs_c2x,-rcs_c2y,0]) {
            BottomChamferedCylinder(radius, height, chamfer);
        }
        translate([-rcs_c2x,-rcs_c2y,0]) {
            BottomChamferedCylinder(radius, height, chamfer);
        }
        translate([-rcs_c2x,rcs_c2y,0]) {
            BottomChamferedCylinder(radius, height, chamfer);
        }
    }
}

module BottomChamferedCylinder(radius, height, chamfer) {
    difference() {
        cylinder(h=height, r=radius);
        difference() {
            translate([0,0,-1]) {
                cylinder(h=chamfer+1, r=radius+1);
            }
            cylinder(h=chamfer*2, r1=radius-chamfer, r2=radius+chamfer);
        }
    }
}

module XYCenteredCube(width, depth, height) {
    translate([-width/2,-depth/2,0]) {
        cube([width, depth, height]);
    }
}